<?php

namespace App\Http\Controllers\Admin;

use Exception;
use App\Models\User;
use App\Models\Booth;
use App\Models\Criteria;
use App\Models\Question;
use App\Models\BoothImage;
use App\Exports\BoothExport;
use App\Imports\BoothImport;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use App\Interfaces\BoothInterface;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Maatwebsite\Excel\Facades\Excel;
use App\Traits\CategorizesBoothRatings;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Response;
use App\Models\Comment;

class BoothController extends Controller
{
    use CategorizesBoothRatings;
    private $booth;
    function __construct(BoothInterface $boothInterface)
    {

        $this->booth = $boothInterface;
    }
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        //
    }
    public function searchBooth(Request $request){

        $booths = $this->booth->getAllBooths($request->search,$request->event_id,$request->hall_id,$request->category);
        return response()->json($booths);
    }

    public function searchBoothByHall(Request $request){
        $hall_id=$request->input('hall_id');
        $event_id = $request->input('event_id');
        $search= $request->input('search');
        $category=$request->input('category');
        $booths = $this->booth->getAllBoothsByHall($hall_id,$event_id,$search, $category);
        return response()->json($booths);
    }
    public function downloadBoothSample(Request $request){
        $filePath = public_path('exports/booths_sample.csv');

        if (!file_exists($filePath)) {
            abort(404);
        }

        return Response::download($filePath, 'sample.csv', [
            'Content-Type' => 'text/csv',
        ]);
    }

    public function searchBoothByStand(Request $request){
        $category= $request->input('category');
        $event_id= $request->input('event_id');
        $hall_id=$request->input('hall_id');
        $search= $request->input('search');
        $booths = $this->booth->getAllBoothsByCategory($category, $event_id,$hall_id,$search);
        return response()->json($booths);
    }

    public function export($event_id)
    {
        $boothsCount = Booth::where('event_id', $event_id)->count();

        if ($boothsCount === 0) {
            return redirect()->back()->with('boothError', 'There are no booths associated with the given event.');
        }

        $export = new BoothExport($event_id);
        return $export->export();
    }

    public function downloadSample()
    {
        $filePath = public_path('exports/sample.csv');

        if (!file_exists($filePath)) {
            abort(404);
        }

        return Response::download($filePath, 'sample.csv', [
            'Content-Type' => 'text/csv',
        ]);
    }

    public function import(Request $request){

        $request->validate([
            'file' => 'required|mimes:csv,txt',
            'event_id' => 'required|exists:events,id',
        ]);

        try {
            $event_id=$request->input('event_id');
            $import = new BoothImport($event_id);
            Excel::import($import, $request->file('file'));

            if ($import->getErrors()->isEmpty()) {
                return back()->with('success', 'Stands imported successfully.');
            } else {
                return back()->with('boothErrors',$import->getErrors()->all())->withInput();
            }
        } catch (Exception $e) {
            //dd($e->getMessage(),$e->getLine(),$e->getFile());
            Log::error('CSV Import Failed.Please select correct file and try again:', ['exception' => $e->getMessage()]);
            return redirect()->back()->with('failed','CSV import failed.Please select correct file and try again');
        }
    }

    public function showDetail(string $id)
    {
        try{
            $booth = Booth::with(['event.users', 'rating.question', 'comment'])->find($id);
            $comment = Comment::where('booth_id', $id)->first();
            $ratings = null;
            $adminEdit = $booth->admin_edit;
            $submitted = $booth->hasUserRating() && (!$booth->draft || $adminEdit);
            $draft = $booth->draft;

            if (($booth->hasUserRating()) && (Auth::user()->getRoleNames()->first() == 'Admin' || Auth::user()->getRoleNames()->first() == 'Super Admin')) {
                $submitted = true;
            }

            if (Auth::user()->getRoleNames()->first() == 'Organizer') {
                $submitted = true;
            }
            $rating = ($submitted || $draft) ? $this->determineCategory($booth->userRating, true) : ['Nill'];
            if (Auth::user()->getRoleNames()->first() != 'Admin' || Auth::user()->getRoleNames()->first() != 'Super Admin') {
                $rating = $this->determineCategory($booth->rating, true);
            }
            //return $submitted;
            $rating = $rating[0];

            $questions = Question::with('rating')->get();

            $questions = Question::with(['rating' => function ($query) use ($id) {
                $query->where('booth_id', $id);
            }])->get();

            $images = BoothImage::where('booth_id', $id)->get();

            $assessmentByUser = null;

            if ($booth->assesment_by) {
                $assessmentByUser = User::find($booth->assesment_by)->name;
            }

            return view('admin.events.booth-detail', ['comment' => $comment, 'assessmentByUser' => $assessmentByUser, 'adminEdit' => $adminEdit, 'draft' => $draft, 'images' => $images, 'booth' => $booth, 'questions' => $questions, 'submitted' => $submitted, 'rating' => $rating]);

        }catch(Exception $ex){
            Log::info("Error in booth detail page. error:".$ex->getMessage());
            return redirect()->back()->with('failed','Somwthing went wrong');
        }
    }

    public function allowEdit(string $id){
        try{
            $booth = Booth::find($id);
            $booth->draft = true;
            $booth->update();
            return redirect()->back()->with('success', 'Stand saved in draft. User can edit and submit again');
        }catch(Exception $ex){
            return redirect()->back()->with('error', 'Some error occured');
        }


    }
    public function allowAdminEdit(string $id)
    {
        try {
            $booth = Booth::find($id);
            $booth->admin_edit = true;
            $booth->update();
            return redirect()->back()->with('success', 'Stand saved in draft for admins. Admins can edit and submit again');
        } catch (Exception $ex) {
            return redirect()->back()->with('error', 'Some error occured');
        }
    }
    public function getRating(Request $request){

        $ratings=$request->input('ratings');
        $rating = $this->determineCategory($ratings);
        return $rating;

    }
    public function storeDetail(Request $request,$id)
    {
        $request->validate([
            'images.*' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:5120', // Max size 5MB
        ]);

        try{

            $newImageNames = $request->input('new-image-names', []);
            $imageNames = $request->input('image-names', []);

            $isDraft = $request->input('isdraft', "0");
            $comment=$request->input('comment',null);
            $ratingSet=$this->booth->updateRating($request->except(['_token', 'isdraft', 'images', 'removedImageIds', 'assesment_by', 'removed_images', 'image-names', 'new-image-names','comment']), $id);
            //dd($request->all(), $ratingSet);
            $removedImages = $request->input('removed_images');


            if ($removedImages) {

                $removedImageIds = explode(',', $removedImages);

                foreach ($removedImageIds as $imageid) {
                    $image = BoothImage::find($imageid);
                    //return $image;
                    if ($image) {
                        // Delete image from storage
                        Storage::disk('public')->delete($image->file);
                        $image->delete();
                    }
                }

                // Optionally, you can perform any additional logic here based on $removedImageIds
            }

            foreach ($imageNames as $iid => $newName) {
                $image = BoothImage::find($iid);
                if ($image) {
                    $image->name = $newName;
                    $image->save();
                }
            }

            // Delete existing images from storage and database

            if ($request->hasfile('images')) {
                //dd($request->file('images'));
                $booth=Booth::find($id);
                $eventName=$booth->event->name;
                $imageCount=BoothImage::where('booth_id',$id)->count();
                foreach ($request->file('images') as $index=>$file) {
                    $imageCount++;
                    $fileNameTemp= time().'_'.'booth_image_' . $imageCount;
                    $filename = $fileNameTemp.'.'. $file->getClientOriginalExtension();
                    $path = Storage::disk('public')->putFileAs('images', $file, $filename);
                   if(isset($newImageNames[$index])){
                        $newName= $newImageNames[$index];
                   }
                   else{
                    $newName= $fileNameTemp;
                   }
                    BoothImage::create(['file' => 'images/' . $filename, 'name' => $newName , 'booth_id' => $id]);
                }
            }


            //$rating = $this->determineCategory($ratings);
            $booth = Booth::find($id);
            //return $booth;
            $cat = $this->determineCategory($booth->userRating, true);
            //return strtoupper($cat[0]);
            $booth->result = strtoupper($cat[0]);
            if(($isDraft == "1")&&(Auth::user()->getRoleNames()->first() == 'Admin' || Auth::user()->getRoleNames()->first() == 'Super Admin')){
                $booth->admin_edit = true;
            }
            else{
                $booth->admin_edit = false;
            }


            $booth->assesment_by = $request->input('assesment_by');
            if ($isDraft == "1") {
                $booth->draft = true;
            } elseif ($isDraft == "0") {
                $booth->draft = false;
            }

            $booth->update();
            if($comment || $comment==''){

                if (Comment::where('booth_id',
                    $id
                )->exists()) {
                    Comment::where('booth_id',$id)->delete();
                }
                $commentObject=new Comment();
                $commentObject->booth_id=$id;
                $commentObject->comment=$comment;
                $booth->comment()->save($commentObject);

            }

            //dd("skip");
            return redirect()->back()->with('success', 'Rating updated successfully');
        }catch(Exception $ex){
            Log::info("error while update rating.".$ex->getMessage());
            return redirect()->back()->with('error', 'Some error occured');
        }
    }
    public function boothStats(Request $request){

    }


    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        //return $request->all();
        $request->validate([
            'number' => [
                'required',
                Rule::unique('booths')->where(function ($query) use ($request) {
                    return $query->where('event_id', $request->event_id);
                }),
            ],
            'event_id' => 'required|exists:App\Models\Event,id',
            'stand_type_id'=>'required',
            'contractor_email'=>'email|nullable',


        ]);

        $registered = $this->booth->createBooth($request->all());

        if ($registered) {
            return redirect()->back()->with('success', 'Stand Registered Successfully');
        } else {
            return  redirect()->back()->withInput()->with(['failed' => 'Failed to register Stand']);
        }

    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $booth = Booth::find($id);

        if (!$booth) {
            return redirect()->back()->with('error', 'Booth not found.');
        }

        // Soft delete the booth
        $booth->delete();

        return redirect()->back()->with('success', 'Booth has been deleted successfully.');
    }
}
